/**
 * BuddyBoss Complete Analytics - Enhanced JavaScript
 */

(function($) {
    'use strict';
    
    let charts = {};
    let currentDateRange = '3 months';
    
    $(document).ready(function() {
        initializeTabs();
        initializeCharts();
        initializeDateRangeSelector();
        initializeRefreshButton();
    });
    
    /**
     * Initialize tab functionality
     */
    function initializeTabs() {
        $('.nav-tab').on('click', function(e) {
            e.preventDefault();
            
            const targetTab = $(this).attr('href');
            
            // Update active tab
            $('.nav-tab').removeClass('nav-tab-active');
            $(this).addClass('nav-tab-active');
            
            // Update active content
            $('.bb-tab-content').removeClass('active');
            $(targetTab).addClass('active');
            
            // Resize charts in the active tab
            setTimeout(function() {
                resizeChartsInActiveTab();
            }, 100);
        });
    }
    
    /**
     * Initialize all charts
     */
    function initializeCharts() {
        if (typeof bbAnalyticsData !== 'undefined' && bbAnalyticsData.data) {
            initializeMembersGrowthChart();
            initializeMemberTypesChart();
            initializeActivityChart();
            initializeActivityTypesChart();
        }
    }
    
    /**
     * Initialize members growth chart
     */
    function initializeMembersGrowthChart() {
        const ctx = document.getElementById('membersGrowthChart');
        if (!ctx || !bbAnalyticsData.data.membersGrowth) return;
        
        const data = bbAnalyticsData.data.membersGrowth;
        const labels = data.map(item => item.month);
        const newMembers = data.map(item => parseInt(item.new_members));
        const totalMembers = data.map(item => parseInt(item.total_members));
        
        charts.membersGrowth = new Chart(ctx, {
            type: 'line',
            data: {
                labels: labels,
                datasets: [{
                    label: 'New Members',
                    data: newMembers,
                    borderColor: '#2271b1',
                    backgroundColor: 'rgba(34, 113, 177, 0.1)',
                    tension: 0.4,
                    yAxisID: 'y'
                }, {
                    label: 'Total Members',
                    data: totalMembers,
                    borderColor: '#00a32a',
                    backgroundColor: 'rgba(0, 163, 42, 0.1)',
                    tension: 0.4,
                    yAxisID: 'y1'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Member Growth Over Time'
                    },
                    legend: {
                        position: 'top'
                    }
                },
                scales: {
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: {
                            display: true,
                            text: 'New Members'
                        }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: {
                            display: true,
                            text: 'Total Members'
                        },
                        grid: {
                            drawOnChartArea: false,
                        },
                    }
                }
            }
        });
    }
    
    /**
     * Initialize member types chart
     */
    function initializeMemberTypesChart() {
        const ctx = document.getElementById('memberTypesChart');
        if (!ctx || !bbAnalyticsData.data.memberTypes) return;
        
        const data = bbAnalyticsData.data.memberTypes;
        const labels = data.map(item => item.member_type);
        const counts = data.map(item => parseInt(item.count));
        
        const colors = [
            '#2271b1', '#00a32a', '#d63638', '#dba617',
            '#8b5cf6', '#f59e0b', '#ef4444', '#10b981',
            '#3b82f6', '#8b5a2b'
        ];
        
        charts.memberTypes = new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: labels,
                datasets: [{
                    data: counts,
                    backgroundColor: colors.slice(0, labels.length),
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Member Types Distribution'
                    },
                    legend: {
                        position: 'right'
                    }
                }
            }
        });
    }
    
    /**
     * Initialize activity over time chart
     */
    function initializeActivityChart() {
        const ctx = document.getElementById('activityChart');
        if (!ctx || !bbAnalyticsData.data.activityOverTime) return;
        
        const data = bbAnalyticsData.data.activityOverTime;
        const labels = data.map(item => item.date);
        const counts = data.map(item => parseInt(item.count));
        
        charts.activity = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Activities',
                    data: counts,
                    backgroundColor: 'rgba(34, 113, 177, 0.8)',
                    borderColor: '#2271b1',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Daily Activity (Last 30 Days)'
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Number of Activities'
                        }
                    },
                    x: {
                        title: {
                            display: true,
                            text: 'Date'
                        }
                    }
                }
            }
        });
    }
    
    /**
     * Initialize activity types chart
     */
    function initializeActivityTypesChart() {
        const ctx = document.getElementById('activityTypesChart');
        if (!ctx || !bbAnalyticsData.data.activityTypes) return;
        
        const data = bbAnalyticsData.data.activityTypes;
        const labels = data.map(item => formatActivityType(item.type));
        const counts = data.map(item => parseInt(item.count));
        
        const colors = [
            '#2271b1', '#00a32a', '#d63638', '#dba617',
            '#8b5cf6', '#f59e0b', '#ef4444', '#10b981'
        ];
        
        charts.activityTypes = new Chart(ctx, {
            type: 'horizontalBar',
            data: {
                labels: labels,
                datasets: [{
                    label: 'Count',
                    data: counts,
                    backgroundColor: colors.slice(0, labels.length),
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Activity Types Distribution'
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Number of Activities'
                        }
                    }
                }
            }
        });
    }
    
    /**
     * Initialize date range selector
     */
    function initializeDateRangeSelector() {
        $('#bb-date-range').on('change', function() {
            const newRange = $(this).val();
            if (newRange !== currentDateRange) {
                currentDateRange = newRange;
                refreshData();
            }
        });
    }
    
    /**
     * Initialize refresh functionality
     */
    function initializeRefreshButton() {
        // Add refresh button to the header
        if ($('.bb-refresh-button').length === 0) {
            $('.bb-analytics-wrap h1').after(
                '<button type="button" class="button button-secondary bb-refresh-button" style="margin-left: 15px;">' +
                '<span class="dashicons dashicons-update"></span> Refresh Data' +
                '</button>'
            );
        }
        
        $('.bb-refresh-button').on('click', function() {
            refreshData();
        });
    }
    
    /**
     * Refresh analytics data
     */
    function refreshData() {
        const $button = $('.bb-refresh-button');
        const originalText = $button.html();
        
        $button.html('<span class="dashicons dashicons-update bb-spin"></span> Refreshing...').prop('disabled', true);
        
        // Clear transients and reload data
        $.post(bbAnalyticsData.ajaxurl, {
            action: 'bb_analytics_refresh_data',
            date_range: currentDateRange,
            nonce: bbAnalyticsData.nonce
        })
        .done(function(response) {
            if (response.success) {
                // Update charts with new data
                updateChartsWithNewData(response.data);
                showNotice('Data refreshed successfully!', 'success');
            } else {
                showNotice('Failed to refresh data. Please try again.', 'error');
            }
        })
        .fail(function() {
            showNotice('Network error. Please check your connection and try again.', 'error');
        })
        .always(function() {
            $button.html(originalText).prop('disabled', false);
        });
    }
    
    /**
     * Update charts with new data
     */
    function updateChartsWithNewData(newData) {
        // Update members growth chart
        if (charts.membersGrowth && newData.membersGrowth) {
            const data = newData.membersGrowth;
            charts.membersGrowth.data.labels = data.map(item => item.month);
            charts.membersGrowth.data.datasets[0].data = data.map(item => parseInt(item.new_members));
            charts.membersGrowth.data.datasets[1].data = data.map(item => parseInt(item.total_members));
            charts.membersGrowth.update();
        }
        
        // Update other charts similarly...
        // (Implementation would continue for other chart types)
    }
    
    /**
     * Resize charts in active tab
     */
    function resizeChartsInActiveTab() {
        Object.keys(charts).forEach(function(chartKey) {
            if (charts[chartKey]) {
                charts[chartKey].resize();
            }
        });
    }
    
    /**
     * Format activity type for display
     */
    function formatActivityType(type) {
        const formatted = {
            'activity_update': 'Status Updates',
            'activity_comment': 'Comments',
            'friendship_accepted': 'Friendships',
            'friendship_created': 'Friend Requests',
            'created_group': 'Group Created',
            'joined_group': 'Group Joined',
            'group_details_updated': 'Group Updated',
            'new_blog_post': 'Blog Posts',
            'new_blog_comment': 'Blog Comments',
            'new_forum_topic': 'Forum Topics',
            'new_forum_post': 'Forum Posts'
        };
        
        return formatted[type] || type.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
    }
    
    /**
     * Show notification
     */
    function showNotice(message, type) {
        const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        const $notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
        
        $('.bb-analytics-wrap').prepend($notice);
        
        setTimeout(function() {
            $notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    /**
     * Add spinning animation for refresh button
     */
    $('<style>')
        .prop('type', 'text/css')
        .html(`
            .bb-spin {
                animation: bb-spin 1s linear infinite;
            }
            @keyframes bb-spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
        `)
        .appendTo('head');
    
    // Export for global access if needed
    window.BuddyBossAnalytics = {
        charts: charts,
        refreshData: refreshData,
        currentDateRange: currentDateRange
    };
    
})(jQuery);

